#R version- R-4.5.0
# Load required libraries
library(caret)
library(mlbench) # for the sample dataset
library(ggplot2)

# Load the dataset
data(Sonar)

# Create training and testing datasets
set.seed(123)
trainIndex <- createDataPartition(Sonar$Class, p = .8, 
                                  list = FALSE, 
                                  times = 1)
data_train <- Sonar[ trainIndex,]
data_test  <- Sonar[-trainIndex,]

# Define control parameters for the model
ctrl <- trainControl(method = "repeatedcv", 
                     number = 10, 
                     repeats = 3, 
                     verboseIter = FALSE)

# Train Random Forest model
set.seed(123)
rf_model <- train(Class ~ ., data = data_train, 
                  method = "rf", 
                  trControl = ctrl)

# Train Gradient Boosting Machine (GBM) model
set.seed(123)
gbm_model <- train(Class ~ ., data = data_train, 
                   method = "gbm", 
                   trControl = ctrl, 
                   verbose = FALSE)

# Train Support Vector Machine (SVM) model
set.seed(123)
svm_model <- train(Class ~ ., data = data_train, 
                   method = "svmRadial", 
                   trControl = ctrl)

# Make predictions
rf_predictions <- predict(rf_model, data_test)
gbm_predictions <- predict(gbm_model, data_test)
svm_predictions <- predict(svm_model, data_test)

# Evaluate the models
rf_accuracy <- confusionMatrix(rf_predictions, data_test$Class)$overall["Accuracy"]
gbm_accuracy <- confusionMatrix(gbm_predictions, data_test$Class)$overall["Accuracy"]
svm_accuracy <- confusionMatrix(svm_predictions, data_test$Class)$overall["Accuracy"]

# Print the accuracies
print(paste("Random Forest Accuracy:", rf_accuracy))
print(paste("Gradient Boosting Machine Accuracy:", gbm_accuracy))
print(paste("Support Vector Machine Accuracy:", svm_accuracy))

# Create a plot comparing model accuracies
model_accuracies <- data.frame(Model = c("Random Forest", "GBM", "SVM"),
                               Accuracy = c(rf_accuracy, gbm_accuracy, svm_accuracy))
ggplot(model_accuracies, aes(x = Model, y = Accuracy, fill = Model)) +
  geom_bar(stat = "identity") +
  ylim(0, 1) +
  labs(title = "Model Accuracies",
       y = "Accuracy",
       x = "Model") +
  theme_minimal()
